<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProfilDesa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class StafDesaController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $profil = ProfilDesa::where('is_active', true)->first();
        $staf = $profil?->staf_tambahan ?? [];
        return view('admin.staf.index', compact('profil', 'staf'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.staf.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $profil = ProfilDesa::where('is_active', true)->firstOrFail();
        $staf = $profil->staf_tambahan ?? [];

        if ($request->hasFile('foto')) {
            $validated['foto'] = $request->file('foto')->store('profil-desa/staf', 'public');
        }

        $staf[] = $validated;
        $profil->update(['staf_tambahan' => $staf]);

        return redirect()->route('admin.staf.index')->with('success', 'Staf desa berhasil ditambahkan!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(int $index)
    {
        $profil = ProfilDesa::where('is_active', true)->firstOrFail();
        $staf = $profil->staf_tambahan ?? [];
        abort_unless(isset($staf[$index]), 404);
        $item = $staf[$index];
        return view('admin.staf.edit', compact('item', 'index'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $index)
    {
        $validated = $request->validate([
            'nama' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $profil = ProfilDesa::where('is_active', true)->firstOrFail();
        $staf = $profil->staf_tambahan ?? [];
        abort_unless(isset($staf[$index]), 404);

        // handle photo replacement
        if ($request->hasFile('foto')) {
            if (!empty($staf[$index]['foto'])) {
                Storage::disk('public')->delete($staf[$index]['foto']);
            }
            $validated['foto'] = $request->file('foto')->store('profil-desa/staf', 'public');
        } else {
            // keep existing path if not replaced
            if (!empty($staf[$index]['foto'])) {
                $validated['foto'] = $staf[$index]['foto'];
            }
        }

        $staf[$index] = $validated;
        $profil->update(['staf_tambahan' => $staf]);

        return redirect()->route('admin.staf.index')->with('success', 'Data staf desa berhasil diperbarui!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $index)
    {
        $profil = ProfilDesa::where('is_active', true)->firstOrFail();
        $staf = $profil->staf_tambahan ?? [];
        abort_unless(isset($staf[$index]), 404);

        if (!empty($staf[$index]['foto'])) {
            Storage::disk('public')->delete($staf[$index]['foto']);
        }

        array_splice($staf, $index, 1);
        $profil->update(['staf_tambahan' => $staf]);

        return redirect()->route('admin.staf.index')->with('success', 'Staf desa berhasil dihapus!');
    }
}